/*
 * (c) rasca, berlin 2004
 * published under the GNU GPL
 */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <getopt.h>
#include <sys/stat.h>
#include <dirent.h>
#include <SDL/SDL.h>
#include <SDL/SDL_image.h>
#include <SDL_rotozoom.h>
#ifndef WIN32
#include <X11/Xlib.h>
#else
#include <windows.h>
#endif

#include "SDL_ttf.h"
#include "help.xpm"
#include "usage.xpm"

#define DEF_WIDTH 1024
#define DEF_HEIGHT 768
#define DEF_TIME 8
#ifdef WIN32
#define PATHSEP '\\'
#else
#define PATHSEP '/'
#endif
#define FONT "cmsr.ttf"
#define FONT_SIZE 16
#define MAXSTRING 1024

#define max(a,b) ((a)>(b)?(a):(b))
#define min(a,b) ((a)<(b)?(a):(b))

typedef struct _file_t {
	char name[PATH_MAX+FILENAME_MAX+1];
	SDL_Surface *image;
	SDL_Surface *thn;	/* thumbnail */
	struct _file_t *next;
	struct _file_t *prev;
} file_t;

/*
 * not used until now
 */
file_t *
file_new (void)
{
	file_t *file = NULL;
	file = malloc (sizeof(file_t));
	file->image = NULL;
	file->thn = NULL;
	file->next = NULL;
	file->prev = NULL;
	return file;
}

/*
 * not used until now
 */
void
file_free (file_t *file)
{
	if (file) {
		free (file);
	}
}

/*
 * try to read the image and create a thumbnail
 * todo..
 */
file_t *
file_thn (char *filename)
{
	return NULL;
}

/*
 */
SDL_Surface *
init_video (int *width, int *height, int bpp, int vflags)
{
	SDL_Rect **modes;
	SDL_Surface *screen;
	// int i;

	if (SDL_Init (SDL_INIT_VIDEO|SDL_INIT_TIMER) == -1) {
		fprintf (stderr, "SDL_Init() failed!\n");
		exit (1);
	}

	/* find a good resolution
	 */
	modes = SDL_ListModes (NULL, vflags);
	if (modes == NULL) {
		fprintf (stderr, "No modes available! (flags=%d)\n", vflags);
		exit (1);
	}
	if (!*width || !*height) {
#ifndef WIN32
	/* get current width and height of the display under X11
	 */
		Display *disp;
		disp = XOpenDisplay (NULL);
		if (disp) {
			*width = DisplayWidth (disp, DefaultScreen (disp));
			*height= DisplayHeight(disp, DefaultScreen (disp));
		}
#else /* windows systems */
		HWND hwnd;
		RECT r;
		/* is there a more elegant way? */
		hwnd = GetDesktopWindow ();
		if (hwnd) {
			if (GetWindowRect (hwnd, &r)) {
				//printf ("GetWindowRect() w=%d h=%d\n", r.right, r.bottom);
				*width = r.right;
				*height= r.bottom;
			}
		}
#endif
	}
	if (modes == (SDL_Rect **)-1) {
		/* all resolutions are ok ..!? */
		if (!*width || !*height) {
			*width = DEF_WIDTH;
			*height= DEF_HEIGHT;
		}
	} else {
		int i;
		for (i = 0; modes[i]; ++i) {
			printf (" %d x %d\n", modes[i]->w, modes[i]->h);
		}
	
		if (!*width || !*height) {
			*width = modes[0]->w;
			*height= modes[0]->h;
		}
	}
#ifdef DEBUG
	printf ("%d x %d\n", *width, *height);
#endif
	screen = SDL_SetVideoMode (*width, *height, bpp, vflags);
	if (!screen) {
		fprintf (stderr, "Video init failed: %s\n", SDL_GetError());
	}
	return screen;
}

void
usage (char *app)
{
	fprintf (stderr, "Usage: %s [-v] [-s] [-F] [-g WxH ] [-d #] <files..>\n",
		app);
	exit (1);
}

int
is_dir (char *file)
{
	struct stat buf;

	if (stat (file, &buf) == 0) {
		if (S_ISDIR(buf.st_mode))
			return 1;
	}
	return 0;
}

void
free_image (file_t* files, int num) {
	if (num < 0)
		return;
	if (!files[num].image)
		return;
	SDL_FreeSurface (files[num].image);
	files[num].image = NULL;
}

#ifdef HAVE_FREETYPE
static TTF_Font *font = NULL;
#endif

/*
 */
SDL_Surface *
load_image (file_t *file, int width, int height, double angle, int scale_up)
{
#ifdef HAVE_FREETYPE
	SDL_Color fg = {0xff, 0xff, 0xff, 0};
#endif
	SDL_Surface *image = NULL;
	int dx, dy, iw, ih;
	double xs, ys;

	file->image = IMG_Load(file->name);
	if (!file->image) {
#ifdef HAVE_FREETYPE
		char s[2*MAXSTRING+1];
		if (strlen (file->name) < MAXSTRING) {
			sprintf (s, "%s: can't open file! (%s)", file->name,SDL_GetError());
			file->image = TTF_RenderText_Blended (font, s, fg);
		} else {
			perror (file->name);
			return NULL;
		}
#else
		perror (file->name);
		return NULL;
#endif
	}
	if ((angle == 90.0) || (angle == 270.0) || (angle == -90.0) ||
					(angle == -270.0)) {
		iw = file->image->h;
		ih = file->image->w;
	} else {
		iw = file->image->w;
		ih = file->image->h;
	}
#ifdef DEBUG
	printf ("DBG: loading %s\n", file->name);
#endif
	if ((iw > width) || (ih > height)) {
		/* scale down */
		dx = iw - width;
		dy = ih - height;
		if (((double)(dx * 3) / (double)4) > dy ) {
			/* x ref */
			xs = ys = (double)width / (double)iw;
		} else {
			/* y ref */
			xs = ys = (double)height / (double)ih;
		}
#ifdef DEBUG
		printf ("DBG: down scaling by %f\n", xs);
#endif
		if (angle != 0) {
			image = rotozoomSurface (file->image, angle, xs, 1);
		} else {
			image = zoomSurface (file->image, xs, ys, 1);
		}
		if (image) {
			SDL_FreeSurface (file->image);
			file->image = image;
		}
	} else if ((scale_up) && ((iw < width) && (ih < height))) {
		/* scale up */
		dx = width - iw;
		dy = height - ih;
		if (((double)(dx * 3) / (double)4) < dy ) {
			/* x ref */
			xs = ys = (double)width / (double)iw;
		} else {
			/* y ref */
			xs = ys = (double)height / (double)ih;
		}
#ifdef DEBUG
		printf ("DBG: up scaling by %f\n", xs);
#endif
		if (angle != 0) {
			image = rotozoomSurface (file->image, angle, xs, 1);
		} else {
			image = zoomSurface (file->image, xs, ys, 1);
		}
		if (image) {
			SDL_FreeSurface (file->image);
			file->image = image;
		}
	}
	return file->image;
}

/*
 */
int
show_filename (file_t *file, SDL_Surface *screen)
{
#ifdef HAVE_FREETYPE
	SDL_Color fg = {0xff, 0xff, 0xff, 0x00};
	SDL_Surface *image;
	image = TTF_RenderText_Blended (font, file->name, fg);
	if (image) {
		SDL_BlitSurface (image, NULL, screen, NULL);
		SDL_UpdateRect (screen, 0, 0, 0, 0);
		SDL_FreeSurface (image);
	} else {
		perror (SDL_GetError());
	}
#else
	printf ("%s\n", file->name);
#endif
	return 0;
}

/*
 */
int
show_image (SDL_Surface *image, SDL_Surface *screen)
{
	SDL_Rect rect;

	if ((image == NULL) || (screen == NULL)) {
		return 0;
	}
	if (image->w < screen->w) {
		rect.x = (screen->w - image->w) / 2;
	} else {
		rect.x = 0;
	}
	if (image->h < screen->h) {
		rect.y = (screen->h - image->h) / 2;
	} else {
		rect.y = 0;
	}
	SDL_FillRect (screen, NULL, 0x00000000);
	SDL_BlitSurface (image, NULL, screen, &rect);
	SDL_UpdateRect (screen, 0, 0, 0, 0);
	return 1;
}

/*
 */
Uint32
tcb_next (Uint32 interval, void *param)
{
	SDL_Event event;

	event.type = SDL_KEYDOWN;
	event.key.keysym.sym = SDLK_n;
	SDL_PushEvent (&event);
	return interval;
}

/*
 */
char *
find_font_file (char *pname, char *font)
{
	char *file = NULL, *p;
	int len;

	if (!pname || !font)
		return NULL;

	if (access (font, R_OK) == 0)
		return strdup (font);

#ifndef WIN32
	len = strlen (pname);
	if (len > sizeof(FONTPATH))
		len = strlen (font) + len + 2;
	else
		len = strlen (font) + sizeof(FONTPATH) + 2;
#else
	len = strlen  (font) + strlen (pname) + 2;
#endif
	
	file = malloc (len);
	sprintf (file, "%s", pname);
	p = strrchr (file, PATHSEP);
	if (!p) {
		free (file);
		return NULL;
	}
	sprintf (p, "%c%s", PATHSEP, font);
	// printf ("(file=%s)\n", file);
	if (access (file, R_OK) == 0)
		return file;
#ifndef WIN32
	sprintf (file, "%s%c%s", FONTPATH, PATHSEP, font);
	// printf ("(file=%s)\n", file);
	if (access (file, R_OK) == 0)
		return file;
#endif
	/* warn */
	return NULL;
}

/*
 */
int
main (int argc, char **argv)
{
	int c, num, i;
	int current;
	int done = 0;
	int verbose = 0;
	int loop = 0;
	int auto_play = 0;
	int scale_up = 0;	/* scale up */
	int bpp = 0;
	int width = 0;
	int height= 0;
	double angle = 0.0;
	SDL_TimerID tid = 0;
	int vflags = SDL_HWSURFACE|SDL_FULLSCREEN;
	SDL_Surface *screen, *image, *help_img;
	SDL_Event event;
	file_t *files = NULL;
	char *font_file = NULL;

	while ((c = getopt (argc, argv, "a:d:g:lsvHFV")) != EOF) {
		switch (c) {
			case 'a':
				auto_play = atoi(optarg) * 1000;
				break;
			case 'd':
				/* display depth */
				bpp = atoi(optarg);
				break;
			case 'g':
				sscanf (optarg, "%dx%d", &width, &height);
				break;
			case 'l':
				loop = 1;
				break;
			case 's':
				scale_up = 1;
				break;
			case 'v':
				verbose++;
				break;
			case 'F':
				vflags &= ~SDL_FULLSCREEN;
				break;
			case 'H':
				vflags &= ~SDL_HWSURFACE;
				break;
			case 'V':
				printf ("XDiaShow, Version %s\n", VERSION);
				return 0;
				break;
			default:
				usage(argv[0]);
				break;
		}
	}
	/* */
#ifdef HAVE_FREETYPE
	TTF_Init();
	font_file = find_font_file (argv[0], FONT);
	if (!font_file) {
		fprintf (stderr, "font file not found: %s\n", FONT);
		return 1;
	}
	font = TTF_OpenFont (font_file, FONT_SIZE);
	if (!font) {
		fprintf (stderr, "%s: %s\n", FONT, SDL_GetError());
		return 1;
	}
#endif

	num = argc - optind;
	if ((num == 1) && is_dir (argv[optind])) {
		DIR *dir;
		struct dirent *en;
		num = 0;
		/* read directory */
		dir = opendir (argv[optind]);
		if (!dir) {
			perror (argv[optind]);
			return 2;
		}
		while ((en = readdir (dir)) != NULL) {
			if (en->d_name[0] == '.')
				continue;
			num++;
		}
		rewinddir (dir);
		files = malloc (sizeof(file_t) * num);
		i = 0;
		while ((en = readdir (dir)) != NULL) {
			if (en->d_name[0] == '.')
				continue;
			sprintf (files[i].name, "%s%c%s", argv[optind], PATHSEP,en->d_name);
			files[i++].image = NULL;
			if (i > num - 1)
				break;
		}
		closedir (dir);
	} else {
		files = malloc (sizeof (file_t) * num);
		for (i = 0; i < num; i++) {
			strcpy (files[i].name, argv[i+optind]);
			files[i].image = NULL;
		}
	}

	image = NULL;
	if (!num) {
		width = 640;
		height= 480;
		vflags &= ~SDL_FULLSCREEN;
		// usage_img = IMG_ReadXPMFromArray (usage_xpm);
		image = IMG_ReadXPMFromArray (usage_xpm);
		//screen = init_video (&width, &height, bpp, vflags);
		/*
		if (screen) {
			show_image (usage_img, screen);
			SDL_Delay (8000);
			return 0;
		} else {
		*/
		if (!image) {
			usage (argv[0]);
		}
	}

	if (verbose > 1) {
		for (i = 0; i < num; i++) {
			printf ("%s\n", files[i].name);
		}
	}
	screen = init_video (&width, &height, bpp, vflags);
	if (!screen) {
		return 2;
	}
	help_img = IMG_ReadXPMFromArray (help_xpm);

	SDL_WM_SetCaption("X-Diashow", "X-Diashow");
	if (vflags & SDL_FULLSCREEN)
		SDL_ShowCursor(0);

	current = 0;
	while (!image && (current < num)) {
		image = load_image (&files[current], width, height, angle, scale_up);
		if (!image)
			current++;
	}
	if (!image) {
		fprintf (stderr, "Can't open any file\n");
		return 2;
	}

	show_image (image, screen);
	if (verbose)
		show_filename (&files[current], screen);

	if (auto_play) {
		tid = SDL_AddTimer (auto_play, tcb_next, NULL);
	}
	while (!done) {
		if (SDL_WaitEvent (&event) > 0) {
			switch (event.type) {
			case SDL_MOUSEBUTTONDOWN:
				if (event.button.button == 1)
					goto PREV;
				if ((event.button.button == 2) || (event.button.button == 3))
					goto NEXT;
				break;
			case SDL_KEYDOWN:
				switch (event.key.keysym.sym) {

				case SDLK_a:
					/* auto play mode */
					if (!tid) {
						if (!auto_play) {
							auto_play = DEF_TIME * 1000;
						}
						tid = SDL_AddTimer (auto_play, tcb_next, NULL);
					} else {
						SDL_RemoveTimer (tid);
						tid = 0;
					}
					break;

				case SDLK_l:
					if (loop)
						loop = 0;
					else
						loop = 1;
					break;

				case SDLK_ESCAPE:
				case SDLK_q:
					done = 1;
					if (vflags & SDL_FULLSCREEN)
						SDL_ShowCursor(1);
					break;

				case SDLK_n:
				case SDLK_RIGHT:
				case SDLK_UP:
					NEXT:
					/* next */
					angle = 0.0;
					image = NULL;
					/* free previous image */
					free_image (files, current - 1);
					if (current < (num - 1)) {
						if (event.key.keysym.sym == SDLK_UP)
							current = min (num - 1, current + 10);
						else
							current++;
						image = files[current].image;
						while (!files[current].image && (current < num)) {
							image = load_image (&files[current],
								width, height, angle, scale_up);
							if (!image)
								current++;
						}
						if (image) {
							show_image (image, screen);
							if (verbose)
								show_filename (&files[current], screen);
						}
					} else if (loop) {
						current = 0;
						image = files[current].image;
						while (!files[current].image && (current < num)) {
							image = load_image (&files[current],
								width, height, angle, scale_up);
							if (!image)
								current++;
						}
						if (image) {
							show_image (image, screen);
							if (verbose)
								show_filename (&files[current], screen);
						}
					}
					// printf ("+ cur = %d\n", current);
					break;

				case SDLK_p:
				case SDLK_LEFT:
				case SDLK_DOWN:
					PREV:
					/* previous */
					angle = 0.0;
					image = NULL;
					if (current > 0) {
						if (event.key.keysym.sym == SDLK_DOWN)
							current = max (0, current-10);
						else
							current--;
						image = files[current].image;
						while (!files[current].image && (current >= 0)) {
							image = load_image (&files[current],
									width, height, angle, scale_up);
							if (!image)
								current--;
						}
						if (image) {
							show_image (image, screen);
							if (verbose)
								show_filename (&files[current], screen);
						}
					}
					// printf ("- cur = %d\n", current);
					break;


				case SDLK_r:
					/* rotate */
					angle -= 90.0;
					if (angle < -270.0)
						angle = 0.0;
					// printf ("new angle: %f\n", angle);
					if (image)
						SDL_FreeSurface (image);
					image = load_image (&files[current],
								width, height, angle, scale_up);
					if (image) {
						show_image (image, screen);
						if (verbose)
							show_filename (&files[current], screen);
					}
					break;

				case SDLK_h:
				case SDLK_F1:
					show_image (help_img, screen);
					break;

				case SDLK_v:
					verbose = verbose ? 0 : 1;
					break;

				case SDLK_PLUS:
				case SDLK_KP_PLUS:
					auto_play += 1000;
					break;

				case SDLK_MINUS:
				case SDLK_KP_MINUS:
					if (auto_play > 1000)
						auto_play -= 1000;
					break;

				case SDLK_f:
					SDL_WM_ToggleFullScreen(screen);
					break;

				default:
					break;
				} /* keysym switch */

				break;

			case SDL_QUIT:
				if (vflags & SDL_FULLSCREEN)
					SDL_ShowCursor(1);
				SDL_Quit();
				return 0;
				break;

			default:
#ifdef DEBUG
				printf ("DBG: ev: %d\n", event.type);
#endif
				break;
			}
		}
	}
	return 0;
}

